import path from 'path';
import { autoUpdater } from 'electron-updater';
import { ipcMain, ipcRenderer } from 'electron';
import { Logger } from './Logger';
import getLicense from './headset/license';

//
// Methods being run from main thread
//
export function setupAutoUpdater(mainWindow) {
    if (process.env.NODE_ENV === 'development') {
        autoUpdater.updateConfigPath = path.join(
            __dirname,
            '../dev-app-update.yml'
        );
    }

    Logger.info('autoUpdater: initializeAutoUpdater');

    autoUpdater.autoDownload = false;
    autoUpdater.logger = Logger;

    const license = getLicense();
    Logger.info('License: ', license);

    if (license.autoUpdateUrl) {
        Logger.info('Setting Auto Update URL: ', license.autoUpdateUrl);
        autoUpdater.setFeedURL(license.autoUpdateUrl);
    }

    // autoUpdater runs in main thread - set it up to receive messages from the renderer thread -
    // specifically checkForUpdates, downloadUpdate and quitAndInstall commands
    ipcMain.on('checkForUpdates', (_event, _props) => {
        try {
            autoUpdater.checkForUpdates();
        } catch (e) {
            Logger.error('autoUpdater error: ', e);
        }
    });
    ipcMain.on('downloadUpdate', (_event, _props) => {
        autoUpdater.downloadUpdate();
    });
    ipcMain.on('quitAndInstallUpdate', (_event, _props) => {
        autoUpdater.quitAndInstall();
    });

    // Send any autoUpdater events from main thread back to renderer thread
    autoUpdater.on('error', err => {
        Logger.error('autoUpdater error: ', err);
        mainWindow.webContents.send('autoUpdateError', err);
    });

    autoUpdater.on('checking-for-update', () => {
        Logger.info('autoUpdater checking for updates');
        mainWindow.webContents.send('autoUpdateCheckingForUpdate');
    });

    autoUpdater.on('update-available', info => {
        Logger.info('autoUpdater - update available: ', info);
        mainWindow.webContents.send('autoUpdateAvailable', info);
    });

    autoUpdater.on('update-not-available', info => {
        Logger.info('autoUpdater - update not available: ', info);
        mainWindow.webContents.send('autoUpdateNotAvailable', info);
    });

    autoUpdater.on('update-downloaded', info => {
        Logger.info('autoUpdater - update downloaded: ', info);
        mainWindow.webContents.send('autoUpdateDownloaded', info);
    });

    autoUpdater.on('download-progress', progressObj => {
        Logger.info(
            `autoUpdater - download progress: ${progressObj.percent}%, ${progressObj.transferred}/ ${progressObj.total} bytes`
        );
        mainWindow.webContents.send('autoUpdateDownloadProgress', progressObj);
    });
}

//
// Methods being run from renderer thread
//
export function checkForUpdates(props) {
    Logger.info('autoUpdater: checkForUpdates');

    // Clean up first
    ipcRenderer.removeAllListeners('autoUpdateError');
    ipcRenderer.removeAllListeners('autoUpdateCheckingForUpdate');
    ipcRenderer.removeAllListeners('autoUpdateAvailable');
    ipcRenderer.removeAllListeners('autoUpdateNotAvailable');
    ipcRenderer.removeAllListeners('autoUpdateDownloaded');
    ipcRenderer.removeAllListeners('autoUpdateDownloadProgress');

    // Prepare - capture all autoUpdater events being sent back from main thread (to our renderer thread)

    ipcRenderer.on('autoUpdateError', (event, data) => {
        if (props.autoUpdateError) {
            props.autoUpdateError(data);
        }
    });

    ipcRenderer.on('autoUpdateCheckingForUpdate', (_event, _data) => {
        if (props.autoUpdateCheckingForUpdate) {
            props.autoUpdateCheckingForUpdate();
        }
    });

    ipcRenderer.on('autoUpdateAvailable', (event, data) => {
        if (props.autoUpdateAvailable) {
            props.autoUpdateAvailable(data);
        }
    });

    ipcRenderer.on('autoUpdateNotAvailable', (event, data) => {
        if (props.autoUpdateNotAvailable) {
            props.autoUpdateNotAvailable(data);
        }
    });

    ipcRenderer.on('autoUpdateDownloaded', (event, data) => {
        if (props.autoUpdateDownloaded) {
            props.autoUpdateDownloaded(data);
        }
    });

    ipcRenderer.on('autoUpdateDownloadProgress', (event, data) => {
        if (props.autoUpdateDownloadProgress) {
            props.autoUpdateDownloadProgress(data);
        }
    });

    // We're in renderer thread, need to send it over to be executed in main thread
    ipcRenderer.send('checkForUpdates');
}

export function downloadUpdate() {
    // We're in renderer thread, need to send it over to be executed in main thread
    ipcRenderer.send('downloadUpdate');
}

export function quitAndInstallUpdate() {
    // We're in renderer thread, need to send it over to be executed in main thread
    ipcRenderer.send('quitAndInstallUpdate');
}
